/**
* \file: signature_db.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <arpa/inet.h>

#include "encryption/signature.h"
#include "model/signature_db.h"
#include "control/configuration.h"
#include "util/helper.h"
#include "ald_types.h"

//---------------------------------- local data structures -----------------------------------------------------------
typedef struct signature_db_entry_t
{
	char *signature;
	char *path;
	signature_db_entry_t *next_entry;
} signature_db_entry_t;

//---------------------------------- local variables -----------------------------------------------------------------
static void *signature_db_data=NULL;
static signature_db_entry_t *first_entry=NULL;
static EVP_PKEY *sig_db_pubkey=NULL;

//---------------------------------- local functions -----------------------------------------------------------------
static error_code_t signature_db_load_db(const char *db_path, size_t sig_db_size);
static error_code_t signature_db_create_linked_list(size_t sig_db_size);
static error_code_t signature_db_add_entry(char *path, char *signature);

static const signature_db_entry_t *signature_db_find_entry(const char *path);

static error_code_t signature_db_verify(const char *db_path);

//---------------------------------- loading and destroy -------------------------------------------------------------
error_code_t signature_db_load(void)
{
	const char *db_path;
	const char *pub_key_path;
	char key_path[PATH_MAX];
	error_code_t result;
	size_t sig_db_size;
	struct stat stat_result;

	if (first_entry!=NULL || signature_db_data!=NULL)
		signature_db_deinit();

	result=signature_init();
	if (result!=RESULT_OK)
		return result;

	db_path=configuration_get_signature_db_path();
	logger_log_debug("SIGNATURE_DB - Loading signature data base from: %s",db_path);

	//the file is one big blob representing the complete db, we are taking the file size to determine the blob size
	if (stat(db_path,&stat_result)==-1)
	{
		logger_log_error("Signature data base not found at \'%s\'- %s.", db_path,strerror(errno));
		return RESULT_FILE_NOT_FOUND;
	}
	sig_db_size=(size_t)stat_result.st_size;

	// read in the public key used to verify the signature db and the scripts
	pub_key_path=configuration_get_signature_db_pub_key_path();
	snprintf(key_path,PATH_MAX,"%s%s",pub_key_path,PUBLIC_KEY_WRP_EXT);
	logger_log_debug("SIGNATURE_DB - Loading public key for the signatures: %s",key_path);

	result=signature_read_public_key(key_path,&sig_db_pubkey, true);
	if (result!=RESULT_OK)
		logger_log_error("Failed to read in the public key to verify the signature data base.\n");

	//verify the data base
	if (result==RESULT_OK)
	{
		logger_log_debug("SIGNATURE_DB - Verifying the signature data base itself.");
		result=signature_db_verify(db_path);
		if (result!=RESULT_OK)
			logger_log_error("Signature data base could not be verified.");
	}

	//in case of an empty db, which we verified successfully, we are done at this point because we do not need to read
	//in something that is not there
	if (result==RESULT_OK && sig_db_size==0)
		return RESULT_OK;

	if (result==RESULT_OK)
		result=signature_db_load_db(db_path,sig_db_size);

	if (result==RESULT_OK)
		result=signature_db_create_linked_list(sig_db_size);

	if (result!=RESULT_OK)
		signature_db_deinit();

	return result;
}

void signature_db_deinit(void)
{
	signature_db_entry_t *entry=first_entry;

	while (entry!=NULL)
	{
		//dequeue entry
		first_entry=entry->next_entry;
		//free it
		free(entry);
		//go to the new first one
		entry=first_entry;
	}

	if (signature_db_data!=NULL)
	{
		free(signature_db_data);
		signature_db_data=NULL;
	}

	signature_destroy_key(sig_db_pubkey);

	signature_deinit();

	sig_db_pubkey=NULL;
}

static error_code_t signature_db_load_db(const char *db_path, size_t sig_db_size)
{
	int db_fd;
	error_code_t result=RESULT_OK;

	db_fd=open(db_path,O_RDONLY);
	if (db_fd==-1)
	{
		logger_log_error("Unable to read signature data base not found at \'%s\'.", db_path);
		result=RESULT_SIGNATURE_DB_INVALID;
	}

	if (result==RESULT_OK)
	{
		signature_db_data=malloc(sig_db_size);
		if (signature_db_data==NULL)
			result=RESULT_NORESOURCE;
	}

	if (result==RESULT_OK)
	{
		if (read(db_fd,signature_db_data,sig_db_size)!=(ssize_t)sig_db_size)
		{
			logger_log_error("Signature data base corrupt.", db_path);
			result=RESULT_SIGNATURE_DB_INVALID;
		}
	}

	if (db_fd!=-1)
		close(db_fd);

	return result;
}

static error_code_t signature_db_create_linked_list(size_t sig_db_size)
{
	error_code_t result;
	size_t sig_db_processed_size=0;
	serialized_signature_db_entry_t *ser_entry;

	ser_entry=signature_db_data;

	if (sig_db_size<sizeof(serialized_signature_db_entry_t))
	{
		logger_log_error("Incomplete signature db entry detected at the end of the data base.");
		return RESULT_SIGNATURE_DB_INVALID;
	}

	do
	{
		uint32_t path_len;

		path_len=ntohl(ser_entry->path_len);

		//bytes processed after we are done with this entry
		sig_db_processed_size+=sizeof(serialized_signature_db_entry_t)+path_len;

		//sanity check: the entry is incomplete if the number of bytes we processed after we are done with the entry
		//exceeds the blob size
		if (sig_db_processed_size>sig_db_size)
		{
			logger_log_error("Incomplete signature db entry detected at the end of the data base.");
			result=RESULT_SIGNATURE_DB_INVALID;
		}
		else
		{
			result=signature_db_add_entry(ser_entry->path,ser_entry->signature);
			//next entry in db starts after the end of the path data of the current entry
			ser_entry=(serialized_signature_db_entry_t *)((void *)((char *)signature_db_data+sig_db_processed_size));
		}
	}
	while (sig_db_processed_size < sig_db_size && result==RESULT_OK);

	return result;
}

static error_code_t signature_db_add_entry(char *path, char *signature)
{
	signature_db_entry_t *new_entry;
	new_entry=malloc(sizeof(signature_db_entry_t));
	if (new_entry==NULL)
		return RESULT_NORESOURCE;

	//set attributes
	new_entry->path=path;
	new_entry->signature=signature;

	//enqueue the new entry
	new_entry->next_entry=first_entry;
	first_entry=new_entry;

	logger_log_debug("SIGNATURE_DB - Signature found for file \'%s\'.", new_entry->path);

	return RESULT_OK;
}

//--------------------------------------------------------------------------------------------------------------------

//---------------------------------- checking signatures -------------------------------------------------------------
bool signature_db_validate_script(const char *path)
{
	const signature_db_entry_t *entry;
	entry=signature_db_find_entry(path);
	//we accept any script that is not contained in the signature db. The execution asks us before executing a script
	//if we actually know it. If not, it is not executing it.
	if (entry==NULL || sig_db_pubkey==NULL)
		return true;

	return signature_check_file(path, entry->signature,sig_db_pubkey)==RESULT_OK;
}

bool signature_db_is_script_available(const char *path)
{
	return signature_db_find_entry(path)!=NULL;
}

static const signature_db_entry_t *signature_db_find_entry(const char *path)
{
	const signature_db_entry_t *entry;
	entry=first_entry;
	while (entry!=NULL)
	{
		if (strcmp(path, entry->path)==0)
			return entry;
		entry=entry->next_entry;
	}

	return NULL;
}

static error_code_t signature_db_verify(const char *db_path)
{
	char signature[RESPONSE_SIGNATURE_SIZE_USED];
	error_code_t result;
	char sig_path[PATH_MAX];

	snprintf(sig_path, PATH_MAX, "%s%s", db_path,SIGNATURE_EXT);
	result=signature_read_from_file(sig_path,signature);
	if (result==RESULT_OK)
		result=signature_check_file(db_path,signature,sig_db_pubkey);

	return result;
}
//--------------------------------------------------------------------------------------------------------------------
